/*

  SmartClient Ajax RIA system
  Version SNAPSHOT_v15.0d_2025-10-29/LGPL Deployment (2025-10-29)

  Copyright 2000 and beyond Isomorphic Software, Inc. All rights reserved.
  "SmartClient" is a trademark of Isomorphic Software, Inc.

  LICENSE NOTICE
     INSTALLATION OR USE OF THIS SOFTWARE INDICATES YOUR ACCEPTANCE OF
     ISOMORPHIC SOFTWARE LICENSE TERMS. If you have received this file
     without an accompanying Isomorphic Software license file, please
     contact licensing@isomorphic.com for details. Unauthorized copying and
     use of this software is a violation of international copyright law.

  DEVELOPMENT ONLY - DO NOT DEPLOY
     This software is provided for evaluation, training, and development
     purposes only. It may include supplementary components that are not
     licensed for deployment. The separate DEPLOY package for this release
     contains SmartClient components that are licensed for deployment.

  PROPRIETARY & PROTECTED MATERIAL
     This software contains proprietary materials that are protected by
     contract and intellectual property law. You are expressly prohibited
     from attempting to reverse engineer this software or modify this
     software for human readability.

  CONTACT ISOMORPHIC
     For more information regarding license rights and restrictions, or to
     report possible license violations, please contact Isomorphic Software
     by email (licensing@isomorphic.com) or web (www.isomorphic.com).

*/
//> @class LocalDataSource
// DataSource implementation using +link{isc.Offline} for the persistence mechanism:  Loads cacheData 
// automatically at init via Offline.get(), and saves it via Offline.put() every time it's 
// modified.
// <P>
// To seed a LocalDataSource with an initial set of values the first time it is loaded in a browser,
// use +link{LocalDataSource.initialCacheData}.
// <P>
// The <code>isc.Offline</code> class makes use of 
// +externalLink{https://developer.mozilla.org/en-US/docs/Web/API/Window/localStorage,HTML5 localStorage}
// to persist data to a user's browser, and have it be available across page reloads, as long as the
// user doesn't explicitly clear the data from their cache.
// <P>
// Developers should be aware of the following security considerations when using LocalDataSources:
// <ul>
//  <li>Data stored in browser localStorage should not contain sensitive information if the user
//      is accessing the data from a public or shared computer, as other users may be able to access
//      it.<br>
//      If your application has features that store sensitive data in a LocalDataSource you 
//      may want to consider warning your end users that they should not use such features on a 
//      shared machine, and/or give them a way of opting out of those features.</li>
//  <li>Access to browser localStorage is restricted by domain and port, so if a SmartClient 
//      application is to be deployed alongside other untrusted applications on the same server
//      (same host and port - for example on a shared hosting provider), 
//      those applications may be able to access stored data.</li>
//  <li>If +link{localDataSource.jsonDateFormat} is set to <code>"logicalDateConstructor"</code>
//      or <code>"dateConstructor"</code>, stored data will be evaluated as script when the dataSource
//      is initialized. This creates a possibility for script injection in the cases described above
//      (public or shared machine, or multiple applications deployed at the same host and port).</li>
// </ul>
// 
// @visibility localDataSource
//<

isc.defineClass("LocalDataSource", "DataSource").addProperties({

    //> @attr localDataSource.cacheData (Array of Record : null : [IR])
    // If cacheData is specified for a local dataSource, it will be used 
    // (and persisted to Offline storage) instead of loading any previously stored 
    // data from Offline storage.
    // <P>
    // Application logic could conditionally apply cacheData to a LocalDataSource to reset
    // previously stored data in certain conditions.
    // <P>
    // Developers wishing to provide a one-time intialization data set instead of 
    // replacing any stored data for this dataSource can use +link{initialCacheData}.
    //
    // @visibility localDataSource
    //<

    //> @attr localDataSource.initialCacheData (Array of Record : null : [IR])
    // When a localDataSource is loaded for the first time in a browser, the
    // Offline storage for the dataSource will be unpopulated. If <code>initialCacheData</code>
    // is populated it will be used as an initial set of data for the dataSource.
    // <P>
    // Note that if +link{cacheData} is populated, that takes precedence over any
    // specified initialCacheData.
    // 
    // @visibility localDataSource
    //<

    cacheAllData: true, 
    clientOnly: true,

    //> @attr localDataSource.jsonDateFormat (JSONDateFormat : logicalDateString : [IR])
    // Format for encoding dates in JSON when serialized for storage in browser local storage.
    // <P>
    // Note that if <code>jsonDateFormat</code> is set to <code>"dateConstructor"</code> or 
    // <code>"logicalDateConstructor"</code>, stored JSON data will be parsed into live records
    // via +link{JSON.decode()}. This ultimately performs a native evaluation of the stored 
    // JS block, allowing javascript method calls to create dates to be embedded in the data.
    // <P>
    // This can be a security concern in certain cases as arbitrary javascript logic
    // stored under the +link{getOfflineStorageKey(),offline storage key} in browser offline storage
    // will be executed within the scope of the application.<br>
    // Access to browser offline storage is restricted by domain and port so this could pose a 
    // security risk for SmartClient applications deployed alongside other untrusted applications 
    // on the same server (same host and port) - for example applications deployed to a large public
    // hosting service, or for users on a public or shared computer.
    // <P>
    // If <code>jsonDateFormat</code> is set to either of these values a warning will be logged
    // to the developer console by default. This may be suppressed by setting +link{warnOnUnsafeStorage} 
    // to false.
    // <P>
    // This security consideration does not apply to the default <code>jsonDateFormat</code> value 
    // of <code>"logicalDateString"</code>. In that case stored JSON will be decoded via 
    // +link{JSON.decodeSafeWithDates()} which uses safe native browser JSON parsing capabilities, 
    // rather than ever being evaluated.
    //
    //  @visibility localDataSource
    //<
    jsonDateFormat:"logicalDateString",

    // disable to:
    //   - avoid pointless copy for an in-memory dataset
    //   - avoid introduction of async flow in response processing that would break synchronousCallback usage that
    //     is sometimes desireable for LocalDataSources
    copyLocalResults: false,

    //> @attr localDataSource.storageKey              (String : null : [IR]) 
    // Optional explicit key to be used for local storage. See +link{getOfflineStorageKey()} for more details.
    //  @visibility localDataSource
    //<


    //> @type OfflineStorageMode
    // @value "allRecords" All data is saved as one serialized blob under the same +link{localDataSource.getOfflineStorageKey(),offline storage key}
    // @value "eachRecord" Each record is stored as different entry under a storage key calculated via +link{localDataSource.getRecordOfflineStorageKey()}
    //  @visibility localDataSource
    //<

    //> @attr localDataSource.storageMode              (OfflineStorageMode : "allRecords" : [IR]) 
    // The storage mode to use.
    // @visibility localDataSource
    //<
    storageMode: "allRecords",

    //> @method localDataSource.getOfflineStorageKey() 
    // Obtain a key to be used for Offline storage.
    // <P>
    // If an explicit +link{storageKey} has been specified it will be used, otherwise the <code>localDataSource.ID</code> will be used.
    // <P>
    // For +link{type:OfflineStorageMode,storageMode:"allRecords"}, the full set of data will be stored as JSON under this key.
    // <P>
    // For +link{type:OfflineStorageMode,storageMode:"eachRecord"}, a list of primary key field values for each record will be stored
    // under this key, and each record's data will be stored separately in offline storage under a key derived via +link{getRecordOfflineStorageKey()}.
    // 
    // @return (String) LocalDataSource.storageKey with LocalDataSource.ID used 
    //                  if storageKey is unset
    // @visibility localDataSource
    //<
    getOfflineStorageKey : function() {
        return this.storageKey || this.getID();
    },

    //> @method localDataSource.getRecordOfflineStorageKey()
    // For +link{storageMode,storageMode:"eachRecord"}, each record's data will be stored separately in offline storage.
    // <P>
    // Default implementation will return the +link{getOfflineStorageKey(),offlineStorageKey} for the dataSource combined
    // with the primary key field value for the record.
    //
    // @param primaryKey (String) primary key field value of record being stored/retrieved from Offline storage. Note that 
    //  if the primaryKey field value is not a string [integer, say], this will be the toString()'d value of that integer.
    // @visibility localDataSource
    //<
    // Making this a public, overrideable method gives devs a way to avoid potential collisions - for example a dataSources
    // named "supplyItem2" could collide with record with pk value of 2 from dataSource "supplyItem"
    
    getRecordOfflineStorageKey : function (pk) {
        return this.getOfflineStorageKey() + pk;
    },

    // This method will update offline storage to match our current cache data
    
    _resetOfflineStorage : function () {
        if (this.storageMode == "allRecords") {
            isc.Offline.put(this.getOfflineStorageKey(), isc.JSON.encode(this.cacheData, {strictQuoting:true, dateFormat:this.jsonDateFormat}));
        } else if (this.storageMode == "eachRecord") {
            var currentPKs = isc.Offline.get(this.getOfflineStorageKey()),
                cacheData = this.cacheData || [];
            if (currentPKs == null) currentPKs = []
            else currentPKs = currentPKs.split(",");

            var newPKs = "";

            for (var i = 0; i < cacheData.length; i++) {
                var record = cacheData[i],
                    pk = record[this.getPrimaryKeyFieldName()];

                currentPKs.remove(pk);
                newPKs += (i == 0 ? "" : ",") + pk;

                isc.Offline.put(this.getRecordOfflineStorageKey(pk), isc.JSON.encode(record, {strictQuoting:true, dateFormat:this.jsonDateFormat}));
            }
            for (var i = 0; i < currentPKs.length; i++) {
                isc.Offline.put(this.getRecordOfflineStorageKey(currentPKs[i]), null);
            }
            isc.Offline.put(this.getOfflineStorageKey(), newPKs);
        }

    },

    updateOfflineCache : function(dsResponse, dsRequest) {
        if (this.storageMode == "allRecords") {
            
            this.fireOnPause(
                "_updateOfflineStorage",
                {target:this, methodName:"_resetOfflineStorage"},
                1
            );
            // isc.Timer.setTimeout(func, 1);        

        } else if (this.storageMode == "eachRecord") {
            // In eachRecord mode we need to update two things:
            // - the primaryKey array that allows us to find our records in offline storage
            // - the data for each modified record
            var record = dsResponse.data;
            if (isc.isAn.Array(record)) record = record[0];
            var pkVal = record && record[this.getPrimaryKeyFieldName()],
                indexKey = this.getOfflineStorageKey();
            if (dsRequest.operationType == "add") {
                if (pkVal != null) {
                    var pkList = isc.Offline.get(indexKey);
                    if (pkList == "" || pkList == null) pkList = "" + pkVal
                    else pkList += "," + pkVal;
                    isc.Offline.put(indexKey, pkList);

                    isc.Offline.put(this.getRecordOfflineStorageKey("" + pkVal), isc.JSON.encode(record, {strictQuoting:true, dateFormat:this.jsonDateFormat}));
                }
            } else if (dsRequest.operationType == "update") {
                if (record != null) {
                    isc.Offline.put(this.getRecordOfflineStorageKey("" + pkVal), isc.JSON.encode(record, {strictQuoting:true, dateFormat:this.jsonDateFormat}));
                }
            } else if (dsRequest.operationType == "remove") {
                isc.Offline.put(this.getRecordOfflineStorageKey("" + pkVal), null);

                var pkList = isc.Offline.get(indexKey);
                if (pkList == null) pkList = [];
                else pkList = pkList.split(",");
                pkList.remove(pkVal + "");

                isc.Offline.put(indexKey, pkList.join(","));

            } else {
                this.logInfo("DataSource operation of type " + dsRequest.operationType + ". Reinitializing offline storage");
            }

        }
    },

    //> @attr localDataSource.warnOnUnsafeStorage (boolean : true : IR)
    // Should a warning be logged when instantiating a localDataSource with settings that require
    // data stored in browser local storage to be evaluated rather than parsed using +link{JSON.decodeSafe()}.
    // <P>
    // See +link{jsonDateFormat} for more information.
    // @visibility localDataSource
    //< 
    warnOnUnsafeStorage:true,
    logUnsafeStorageWarning : function () {
        if (this.warnOnUnsafeStorage) {
            this.logWarn("LocalDataSource created with specified jsonDateFormat:'" + this.jsonDateFormat + 
                "'. This setting can be a security risk in some configurations. See documentation for details.");
        }
    },

    init : function () {
        this.Super("init", arguments);
        if (this.jsonDateFormat == "logicalDateConstructor" || this.jsonDateFormat == "dateConstructor") {
            this.logUnsafeStorageWarning();
        }
        var data = this.cacheData,
            mustUpdateStorage = (data != null);
        
        if (! data) {
            var hasOfflineData = false;
            if (this.storageMode == "eachRecord" && this.getPrimaryKeyField() == null) {
                this.logWarn("LocalDataSource initialized with storageMode 'eachRecord' has no primary key field. " + 
                    "This is unsupported - switching to 'allRecords' storage mode.");
                this.storageMode = "allRecords"
            }

            var storageKey = this.getOfflineStorageKey();
            if (this.storageMode == "allRecords") {
                var json = isc.Offline.get(storageKey);
                if (json) {
                    // StorageKey arg is just for logging
                    data = this.decodeJSON(json, storageKey);
                }
                if (!isc.isA.Array(data)) {
                    data = [];
                } else {
                    hasOfflineData = true;
                }

            } else {
                var jsonList = isc.Offline.get(storageKey);
                if (jsonList != null) {
                    hasOfflineData = true;
                    var pks = jsonList.split(",");
                    data = [];
                    if (pks.length > 0) {
                        hasOfflineData = true;
                        var loggedWarning = false;
                        for (var i = 0; i < pks.length; i++) {
                            if (pks[i] == null) continue;
                            var recordKey = this.getRecordOfflineStorageKey(pks[i]),
                                recordJSON = isc.Offline.get(recordKey),
                                record = this.decodeJSON(recordJSON, (loggedWarning ? null : recordKey));
                            if (record != null) data[data.length] = record;
                            else loggedWarning = true;
                        }
                    }
                }
            }
            if (!hasOfflineData) {
                data = this.initialCacheData || data;
                mustUpdateStorage = true;
            }

            this.setCacheData(data);
        }
        // If we're populated from explicitly specified cacheData or initialCacheData we need to initialize
        // offline storage.
        if (mustUpdateStorage) {
            this._resetOfflineStorage();
        }

        this.observe(this, "updateCaches", "observer.updateOfflineCache(arguments[0], arguments[1])");
    },

    // Helper to decode json blocks from offline storage.
    // Will return null if it can't find a valid JSON block.
    decodeJSON : function (json, logStorageKey) {

        var data;
        if (this.jsonDateFormat == "logicalDateConstructor" || this.jsonDateFormat == "dateConstructor") {
            this.logUnsafeStorageWarning();
            data = isc.JSON.decode(json, {dateFormat: this.jsonDateFormat});
        } else {
            try {
                if (this.jsonDateFormat == "logicalDateString") {
                    data = isc.JSON.decodeSafeWithDates(json, true);   
                } else {
                    // this.jsonReviver is an undocumented way to handle custom data formats
                    data = isc.JSON.decodeSafe(json, this.jsonReviver, true);
                }
            } catch (e) {
                if (logStorageKey) {
                    this.logWarn("LocalDataSource - unable to parse data stored under:" + logStorageKey);
                    if (this.logIsDebugEnabled()) {
                        this.logDebug("Logging JSON string to browser console");
                        console.warn(json);
                    }
                }
                data = null;
            }
        }
        return data;
    }
});